using UnityEngine;
using UnityEngine.Events;
	
namespace MIGames.Weapons.Charges {
	public class EnergyCharge : ProjectileCharge {
		[Header("Recharge")]
		[Tooltip("Is auto recharging enabled?")]
		public bool rechargeEnabled = false;
		[Tooltip("Timeout after last shot before recharge.")]
		public float rechargeDelay = 1f;
		[Tooltip("Number of charges restored per second.")]
		public float rechargeRate = 32f;

		protected float currentRechargeDelay;

		void Update() {
			if(!this.rechargeEnabled) {
				return;
			}

			if(this.currentRechargeDelay > 0f) {
				this.currentRechargeDelay = Mathf.Max(this.currentRechargeDelay - Time.deltaTime, 0f);
			} else if(this.charges < this.maxCharges && (this.infiniteStorage || this.chargesStored > 0f)) {
				float charge = Time.deltaTime * this.rechargeRate;
				if(!this.infiniteStorage) {
					charge = Mathf.Min(charge, this.chargesStored);
				}
				charge = Mathf.Min(charge, this.maxCharges - this.charges);

				this.charges += charge;
				this.chargesStored -= charge;

				if(charge > 0f && this.onChargeUpdate != null) this.onChargeUpdate.Invoke();
			}
		}
		
		public override bool CanSupply(WeaponTriggerData trigger) {
			return base.CanSupply(trigger) && this.charges >= trigger.charge;
		}

		public override bool TrySupply(WeaponTriggerData trigger, out float charge) {
			if(!CanSupply(trigger)) {
				charge = 0f;
				return false;
			}	

			charge = Mathf.Min(trigger.charge, this.charges);

			this.charges -= charge;
			
			if(this.rechargeEnabled) {
				this.currentRechargeDelay = this.rechargeDelay;
			}

			if(this.onChargeUpdate != null) this.onChargeUpdate.Invoke();

			return true;		
		}
	}
}
