using UnityEngine;
using UnityEngine.Events;
	
namespace MIGames.Weapons.Charges {
	public class ProjectileCharge : MonoBehaviour, WeaponCharge {
		[Header("General")]
		[Tooltip("Max amount of charges.")]
		public float maxCharges = 32f;

		[Header("Storage")]
		[Tooltip("Is charge store infinite?")]
		public bool infiniteStorage = false;
		[Tooltip("Amount of charges in store.")]
		public float chargesStored = 64f;

		[Header("Reload")]
		[Tooltip("Is reloading enabled?")]
		public bool reloadEnabled = true;
		[Tooltip("Is auto reload enabled?")]
		public bool reloadAuto = false;
		[Tooltip("Duration of performing reload.")]
		public float reloadDuration = 1f;
		[Tooltip("Relative time for applying ammo while reloading.")]
		public float reloadApplyTime = 1f;

		protected float charges;
		protected bool reloading;

		protected UnityAction onChargeUpdate;
		protected UnityAction onReloadStart;
		protected UnityAction onReloadEnd;
		protected UnityAction onReloadApply;

		void Start() {
			this.charges = this.maxCharges;
		}

		// --- (un)register event handlers

		public void RegisterChargeUpdateAction(UnityAction action) {
			this.onChargeUpdate += action;
		}

		public void UnregisterChargeUpdateAction(UnityAction action) {
			this.onChargeUpdate -= action;
		}

		public void RegisterReloadStartAction(UnityAction action) {
			this.onReloadStart += action;
		}

		public void UnregisterReloadStartAction(UnityAction action) {
			this.onReloadStart -= action;
		}

		public void RegisterReloadEndAction(UnityAction action) {
			this.onReloadEnd += action;
		}

		public void UnregisterReloadEndAction(UnityAction action) {
			this.onReloadEnd -= action;
		}

		public void RegisterReloadApplyAction(UnityAction action) {
			this.onReloadApply += action;
		}

		public void UnregisterReloadApplyAction(UnityAction action) {
			this.onReloadApply -= action;
		}

		// --- charge implementations
		
		public virtual bool CanSupply(WeaponTriggerData trigger) {
			bool canSupply = this.charges >= 1f && !IsReloading();			
			if(!canSupply && this.reloadAuto && CanReload() && !IsReloading()) {
				TryReload();
			}
			return canSupply;
		}

		public virtual bool TrySupply(WeaponTriggerData trigger, out float charge) {
			if(!CanSupply(trigger)) {
				charge = 0f;
				return false;
			}	

			charge = 1f;
			this.charges -= 1f;

			if(this.onChargeUpdate != null) this.onChargeUpdate.Invoke();

			return true;		
		}

		public float GetCharge() {
			return this.charges / this.maxCharges;
		}

		public void SetCharge(float charge) {
			this.charges = this.maxCharges * charge;
			if(this.onChargeUpdate != null) this.onChargeUpdate.Invoke();
		}

		// --- reload implementations

		public virtual bool CanReload() {
			return this.reloadEnabled && !this.reloading && this.charges < this.maxCharges && (this.infiniteStorage || this.chargesStored > 0f);
		}

		public virtual bool TryReload() {
			if(!CanReload()) {
				return false;
			}

			return ReloadStart();
		}

		public virtual bool IsReloading() {
			return this.reloading;
		}

		public virtual void BreakReload() {
			ReloadEnd();
		}

		// --- specials

		public float GetCurrentCharges() {
			return this.charges;
		}

		protected virtual bool ReloadStart() {
			if(!CanReload()) {
				return false;
			}

			this.reloading = true;

			if(this.onReloadStart != null) this.onReloadStart.Invoke();

			CancelInvoke("ReloadApply");
			CancelInvoke("ReloadEnd");

			Invoke("ReloadApply", this.reloadDuration * Mathf.Clamp01(this.reloadApplyTime));
			Invoke("ReloadEnd", this.reloadDuration);

			return true;
		}

		protected virtual void ReloadApply() {
			if(!this.reloading || this.charges >= this.maxCharges || (!this.infiniteStorage && this.chargesStored <= 0f)) {
				return;
			}

			float recharge = this.maxCharges - this.charges;
			if(!this.infiniteStorage) {
				recharge = Mathf.Min(recharge, this.chargesStored);

				this.chargesStored -= recharge;
			}

			this.charges += recharge;

			if(this.onReloadApply != null) this.onReloadApply.Invoke();
			if(this.onChargeUpdate != null) this.onChargeUpdate.Invoke();

			CancelInvoke("ReloadApply");
		}

		protected virtual void ReloadEnd() {
			if(!this.reloading) {
				return;
			}

			this.reloading = false;

			if(this.onReloadEnd != null) this.onReloadEnd.Invoke();

			CancelInvoke("ReloadApply");
			CancelInvoke("ReloadEnd");
		}
	}
}
