using UnityEngine;

using MIGames.Weapons.Charges;

namespace MIGames.Weapons.FX {
	public class WeaponSFX : MonoBehaviour {
		public AudioSource audioSource;

		[Header("Trigger")]
		[Tooltip("Enable trigger sfx?")]
		public bool triggerEnabled = true;
		public AudioClip triggerSFX;
		public AudioClip triggerStartSFX;
		public AudioClip triggerEndSFX;
		public AudioClip triggerLoopSFX;

		[Header("Action")]
		[Tooltip("Enable action sfx?")]
		public bool actionEnabled = true;
		public AudioClip actionSFX;
		public AudioClip actionStartSFX;
		public AudioClip actionEndSFX;
		public AudioClip actionLoopSFX;

		[Header("Reload")]
		[Tooltip("Enable reload sfx?")]
		public bool reloadEnabled = true;
		public AudioClip reloadStartSFX;
		public AudioClip reloadApplySFX;
		public AudioClip reloadEndSFX;
		public AudioClip reloadLoopSFX;
		public AudioClip reloadStretchSFX;
		public AudioClip reloadStretchApplySFX;

		[Header("Equip")]
		[Tooltip("Enable equip sfx?")]
		public bool equipEnabled = true;
		public AudioClip equipSFX;
		public AudioClip unequipSFX;

		protected WeaponTrigger trigger;
		protected WeaponAction action;
		protected WeaponCharge charge;

		void Start() {
			if(this.triggerEnabled) {
				this.trigger = GetComponent<WeaponTrigger>();
				if(this.trigger != null) {
					this.trigger.RegisterTriggerAction(Trigger);
					this.trigger.RegisterTriggerStartAction(TriggerStart);
					this.trigger.RegisterTriggerEndAction(TriggerEnd);
				}
			}

			if(this.actionEnabled) {
				this.action = GetComponent<WeaponAction>();
				if(this.action != null) {
					this.action.RegisterPerformAction(Action);
					this.action.RegisterPerformStartAction(ActionStart);
					this.action.RegisterPerformEndAction(ActionEnd);
				}
			}

			if(this.reloadEnabled) {
				this.charge = GetComponent<WeaponCharge>();
				if(this.charge != null) {
					this.charge.RegisterReloadStartAction(ReloadStart);
					this.charge.RegisterReloadEndAction(ReloadEnd);
					this.charge.RegisterReloadApplyAction(ReloadApply);
				}
			}

			//TODO: equip/unequip
		}

		// --- event receivers

		void Trigger() {
			Play(this.triggerSFX);
		}

		void TriggerStart() {
			Play(this.triggerStartSFX);
			Loop(this.triggerLoopSFX);
		}

		void TriggerEnd() {
			Play(this.triggerEndSFX);
			LoopEnd();
		}

		void Action() {
			Play(this.actionSFX);
		}

		void ActionStart() {
			Play(this.actionStartSFX);
			Loop(this.actionLoopSFX);
		}

		void ActionEnd() {
			Play(this.actionEndSFX);
			LoopEnd();
		}

		void ReloadStart() {
			Play(this.reloadStartSFX);
			Loop(this.reloadLoopSFX);

			if(this.charge != null && this.charge is ProjectileCharge) {
				ProjectileCharge projectileCharge = (ProjectileCharge)this.charge;
				PlayDuration(this.reloadStretchSFX, projectileCharge.reloadDuration);
				PlayDuration(this.reloadStretchApplySFX, projectileCharge.reloadDuration * projectileCharge.reloadApplyTime);
			}
		}

		void ReloadApply() {
			Play(this.reloadApplySFX);
		}

		void ReloadEnd() {
			Play(this.reloadEndSFX);
			LoopEnd();
		}

		// --- helpers

		protected void Play(AudioClip clip) {
			Play(clip, 1f);
		}

		protected void Play(AudioClip clip, float pitch) {
			if(this.audioSource == null || clip == null) return;
			this.audioSource.pitch = pitch;
        	this.audioSource.PlayOneShot(clip);
		}

		protected void PlayDuration(AudioClip clip, float duration) {
			if(this.audioSource == null || clip == null) return;
			Play(clip, clip.length / duration);
		}

		protected void Loop(AudioClip clip) {
			if(this.audioSource == null || clip == null) return;
        	this.audioSource.clip = clip;
			this.audioSource.loop = true;
			this.audioSource.Play();
		}

		protected void LoopEnd() {
			if(this.audioSource == null || !this.audioSource.loop) {
				return;
			}
			this.audioSource.loop = false;
			this.audioSource.Stop();
		}
	}
}
