﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;

namespace MIGames.Units.Movement {
	public class CharacterControllerMovement : AbstractCharacterMovement {
		public CharacterController controller;

		void Start() {
			if(this.controller == null) this.controller = GetComponent<CharacterController>();
		}

		void FixedUpdate() {
			Process(Time.fixedDeltaTime);
		}

		void OnDeath() {
			this.controller.enabled = false;
		}

		void OnRevive() {
			this.controller.enabled = true;
		}

		public override Vector3 GetVelocity() {
			return new Vector3();
		}

		public override Vector3 GetAngularVelocity() {
			return new Vector3();
		}

		protected virtual float GetGroundCheckDistance() {
		    // Make sure that the ground check distance while already in air is very small, to prevent suddenly snapping to ground
			return this.isGrounded ? (this.controller.skinWidth + groundCheckDistance) : k_GroundCheckDistanceInAir;
		}

		protected override void GroundCheck() {
		    float chosenGroundCheckDistance = GetGroundCheckDistance();

		    // reset values before the ground check
		    isGrounded = false;
		    this.groundNormal = Vector3.up;

		    // only try to detect ground if it's been a short amount of time since last jump; otherwise we may snap to the ground instantly after we try jumping
		    if (Time.time >= m_LastTimeJumped + k_JumpGroundingPreventionTime)
		    {
		        // if we're grounded, collect info about the ground normal with a downward capsule cast representing our character capsule
		        if (Physics.CapsuleCast(GetCapsuleBottomHemisphere(), GetCapsuleTopHemisphere(this.controller.height), this.controller.radius, Vector3.down, out RaycastHit hit, chosenGroundCheckDistance, groundCheckLayers, QueryTriggerInteraction.Ignore))
		        {
		            // storing the upward direction for the surface found
		            this.groundNormal = hit.normal;

		            // Only consider this a valid ground hit if the ground normal goes in the same direction as the character up
		            // and if the slope angle is lower than the character controller's limit
		            if (Vector3.Dot(hit.normal, transform.up) > 0f &&
		                IsNormalUnderSlopeLimit(this.groundNormal))
		            {
		                isGrounded = true;

		                // handle snapping to the ground
		                if (hit.distance > this.controller.skinWidth)
		                {
		                    this.controller.Move(Vector3.down * hit.distance);
		                }
		            }
		        }
		    }
		}

		protected override void HandleCharacterMovement() {
			if(!this.controller.enabled) {
				return;
			}	
	
		    // horizontal character rotation
		    {
		        // rotate the transform with the input speed around its local Y axis
		        //transform.Rotate(new Vector3(0f, (m_InputHandler.GetLookInputsHorizontal() * rotationSpeed * RotationMultiplier), 0f), Space.Self);
				transform.Rotate(this.inputRotate * this.rotationSpeed * this.RotationMultiplier, Space.Self);
		    }

		    // vertical camera rotation
		    //{
		        // add vertical inputs to the camera's vertical angle
		        //m_CameraVerticalAngle += m_InputHandler.GetLookInputsVertical() * rotationSpeed * RotationMultiplier;

		        // limit the camera's vertical angle to min/max
		        //m_CameraVerticalAngle = Mathf.Clamp(m_CameraVerticalAngle, -89f, 89f);

		        // apply the vertical angle as a local rotation to the camera transform along its right axis (makes it pivot up and down)
				//Transform viewTransform = this.playerView != null ? this.playerView : this.playerCamera.transform;
				//viewTransform.localEulerAngles = new Vector3(m_CameraVerticalAngle, 0, 0);
		    //}

		    // character movement handling
		    //bool isSprinting = m_InputHandler.GetSprintInputHeld();
		    {
		        //if (isSprinting)
		        //{
		        //    isSprinting = SetCrouchingState(false, false);
		        //}


		        // converts move input to a worldspace vector based on our character's transform orientation
		        Vector3 worldspaceMoveInput = transform.TransformVector(this.inputMove);
				bool jump = this.inputMove.z > 0f;
		        //float speedModifier = isSprinting ? sprintSpeedModifier : 1f;
				float speedModifier = 1f;

		        // handle grounded movement
		        if(isGrounded) {
		            // calculate the desired velocity from inputs, max speed, and current slope
		            Vector3 targetVelocity = worldspaceMoveInput * maxSpeedOnGround * speedModifier;
		            // reduce speed if crouching by crouch speed ratio
		            //if (isCrouching)
		            //    targetVelocity *= maxSpeedCrouchedRatio;
		            targetVelocity = GetDirectionReorientedOnSlope(targetVelocity.normalized, this.groundNormal) * targetVelocity.magnitude;

		            // smoothly interpolate between our current velocity and the target velocity based on acceleration speed
		            characterVelocity = Vector3.Lerp(characterVelocity, targetVelocity, movementSharpnessOnGround * Time.deltaTime);

		            // jumping
		            if (isGrounded && jump) {
		                // force the crouch state to false
		                //if (SetCrouchingState(false, false)) {
		                    // start by canceling out the vertical component of our velocity
		                    characterVelocity = new Vector3(characterVelocity.x, 0f, characterVelocity.z);

		                    // then, add the jumpSpeed value upwards
		                    characterVelocity += Vector3.up * jumpForce;

		                    // play sound
		                    this.onJump.Invoke(jumpForce);

		                    // remember last time we jumped because we need to prevent snapping to ground for a short time
		                    m_LastTimeJumped = Time.time;
		                    hasJumpedThisFrame = true;

		                    // Force grounding to false
		                    isGrounded = false;
		                    this.groundNormal = Vector3.up;
		                //}
		            }

		            // footsteps sound
		            //float chosenFootstepSFXFrequency = (isSprinting ? footstepSFXFrequencyWhileSprinting : footstepSFXFrequency);
		            if (m_footstepDistanceCounter >= 1f / this.footstepFrequency)
		            {
						this.onStep.Invoke(1f);
		            }

		            // keep track of distance traveled for footsteps sound
		            //m_footstepDistanceCounter += characterVelocity.magnitude * Time.deltaTime;
		        }
		        // handle air movement
		        else {
		            // add air acceleration
		            characterVelocity += worldspaceMoveInput * accelerationSpeedInAir * Time.deltaTime;

		            // limit air speed to a maximum, but only horizontally
		            float verticalVelocity = characterVelocity.y;
		            Vector3 horizontalVelocity = Vector3.ProjectOnPlane(characterVelocity, Vector3.up);
		            horizontalVelocity = Vector3.ClampMagnitude(horizontalVelocity, maxSpeedInAir * speedModifier);
		            characterVelocity = horizontalVelocity + (Vector3.up * verticalVelocity);

		            // apply the gravity to the velocity
		            characterVelocity += Vector3.down * gravityDownForce * Time.deltaTime;
		        }
		    }

		    // apply the final calculated velocity value as a character movement
		    Vector3 capsuleBottomBeforeMove = GetCapsuleBottomHemisphere();
		    Vector3 capsuleTopBeforeMove = GetCapsuleTopHemisphere(this.controller.height);
		    this.controller.Move(characterVelocity * Time.deltaTime);

		    // detect obstructions to adjust velocity accordingly
		    m_LatestImpactSpeed = Vector3.zero;
		    if (Physics.CapsuleCast(capsuleBottomBeforeMove, capsuleTopBeforeMove, this.controller.radius, characterVelocity.normalized, out RaycastHit hit, characterVelocity.magnitude * Time.deltaTime, -1, QueryTriggerInteraction.Ignore)) {
		        // We remember the last impact speed because the fall damage logic might need it
		        m_LatestImpactSpeed = characterVelocity;

		        characterVelocity = Vector3.ProjectOnPlane(characterVelocity, hit.normal);
		    }
		}

		// Returns true if the slope angle represented by the given normal is under the slope angle limit of the character controller
		bool IsNormalUnderSlopeLimit(Vector3 normal) {
		    return Vector3.Angle(transform.up, normal) <= this.controller.slopeLimit;
		}

		// Gets the center point of the bottom hemisphere of the character controller capsule    
		Vector3 GetCapsuleBottomHemisphere() {
		    return transform.position + (transform.up * this.controller.radius);
		}

		// Gets the center point of the top hemisphere of the character controller capsule    
		Vector3 GetCapsuleTopHemisphere(float atHeight) {
		    return transform.position + (transform.up * (atHeight - this.controller.radius));
		}

		// Gets a reoriented direction that is tangent to a given slope
		public Vector3 GetDirectionReorientedOnSlope(Vector3 direction, Vector3 slopeNormal) {
		    Vector3 directionRight = Vector3.Cross(direction, transform.up);
		    return Vector3.Cross(slopeNormal, directionRight).normalized;
		}
	}
}
