using UnityEngine;
using UnityEngine.Events;

using MIGames.Weapons.Handlers;

namespace MIGames.Weapons {
	public class WeaponController : MonoBehaviour {
		public Transform aim;

		public GameObject equippedState;
		public GameObject unequippedState;

		protected WeaponTrigger trigger;
		protected WeaponAction action;
		protected WeaponCharge charge;

		protected UnityAction onTrigger;
		protected UnityAction onHit;
		protected UnityAction onEquip;
		protected UnityAction onUnequip;

		protected bool wasTriggered = false;
		protected bool triggerActive = false;
		protected bool ready = true;

		protected WeaponTriggerData currentTriggerData;
		protected WeaponHitData currentHitData;

		protected WeaponHandler handler;

		// TODO: add validators to allow external scripts to decide if trigger / hit is valid

		void Start() {
			this.trigger = GetComponent<WeaponTrigger>();
			if(this.trigger != null) {
				this.trigger.RegisterTriggerAction(TriggerShot);
				this.trigger.RegisterTriggerStartAction(TriggerShotStart);
				this.trigger.RegisterTriggerEndAction(TriggerShotEnd);
			}

			this.action = GetComponent<WeaponAction>();
			if(this.action != null) this.action.RegisterHitAction(HitShot);
			if(this.action != null && this.aim != null) this.action.SetAim(this.aim);

			this.charge = GetComponent<WeaponCharge>();

			// apply handler state, if not set before
			if(this.handler == null) {
				SetHandler(null);
			}
		}

		// --- Register Event Handlers

		public virtual void RegisterTriggerAction(UnityAction handler) {
			this.onTrigger += handler;
		}

		public virtual void UnregisterTriggerAction(UnityAction handler) {
			this.onTrigger -= handler;
		}

		public virtual void RegisterHitAction(UnityAction handler) {
			this.onHit += handler;
		}

		public virtual void UnregisterHitAction(UnityAction handler) {
			this.onHit -= handler;
		}

		// --- Offer Event Details

		public virtual WeaponTriggerData GetCurrentTriggerData() {
			return this.currentTriggerData;
		}

		public virtual WeaponHitData GetCurrentHitData() {
			return this.currentHitData;
		}

		// --- Receive Events

		void TriggerShot() {
			this.currentTriggerData = this.trigger.GetCurrentTriggerData();
			if(!TryPerformShot()) {
				this.trigger.Break();
			}
		}

		void TriggerShotStart() {
			this.triggerActive = true;
		}

		void TriggerShotEnd() {
			this.triggerActive = false;
		}

		void HitShot() {
			this.currentHitData = this.action.GetCurrentHitData();
			TryProcessHit();
		}
			
		// --- Perform Actions

		protected bool TryPerformShot() {
			// check if action and charge is available, otherwise assume that performing shot succeeds (shots may get handled by external script)
			if(
				(this.charge != null && !this.charge.CanSupply(this.currentTriggerData))
				|| (this.action != null && !this.action.CanPerform(this.currentTriggerData))
			) {
				return false;
			}

			this.wasTriggered = true;
			if(this.onTrigger != null) this.onTrigger.Invoke();
			
			// try to perform action and reduce charge if available, otherwise assume that performing shot succeeded (shots may get handled by external script)
			return (
				(this.action == null || this.action.TryPerform(this.currentTriggerData))
				&& (this.charge == null || this.charge.TrySupply(this.currentTriggerData, out this.currentTriggerData.charge))
			);
		}

		protected bool TryProcessHit() {
			if(this.onHit != null) this.onHit.Invoke();
			
			foreach(WeaponHitReceiver receiver in this.currentHitData.collider.GetComponents<WeaponHitReceiver>()) {
				receiver.RegisterHit(this.currentHitData);
			}

			return true;
		}

		// --- Receive Inputs

		protected float lastInputVal;
		public bool HandleInput(float val) {
			this.trigger.SetWeight(val);

			// translate axis input to button input
			bool result = HandleInput(this.lastInputVal == 0f && val != 0f, val != 0f, this.lastInputVal != 0f && val == 0f);

			this.lastInputVal = val;

			return result;
		}

		public bool HandleInput(bool down, bool held, bool up) {
			if(this.trigger == null) {
				// no trigger defined => try to perform shot on every click
				if(down) return TryPerformShot();
				return false;
			}

			// pass input to trigger
			this.trigger.SetWeight(1f);
			if(down) this.trigger.TriggerStart();
			if(up) this.trigger.TriggerEnd();

			// report to caller if action was performed
			bool result = this.wasTriggered;
			this.wasTriggered = false;
			return result;
		}

		public void SetHandler(WeaponHandler handler) {
			/*if(handler == this.handler) {
				return;
			}*/

			if(this.handler != null) {
				if(this.onUnequip != null) this.onUnequip.Invoke(); 
				if(this.equippedState != null) this.equippedState.SetActive(false);
				if(this.unequippedState != null) this.unequippedState.SetActive(true);
			}

			this.handler = handler;

			if(this.handler != null) {
				if(this.onEquip != null) this.onEquip.Invoke(); 
				if(this.equippedState != null) this.equippedState.SetActive(true);
				if(this.unequippedState != null) this.unequippedState.SetActive(false);
			}
		}

		public void SetAim(Transform aim) {
			this.aim = aim;
			if(this.action != null) this.action.SetAim(aim);
		}

		public void Reload() {
			if(this.charge == null || this.triggerActive) return;
			
			this.charge.TryReload();
		}

		public void SetReady(bool ready, float timer) {
			Invoke(ready ? "SetReady" : "SetLocked", timer);
		}

		public void SetReady(bool ready) {
			this.ready = ready;
			CancelInvoke("SetReady");
			CancelInvoke("SetLocked");
		}

		public void SetReady() {
			SetReady(true);
		}

		public void SetLocked() {
			SetReady(false);
		}
	}
}
